/*
 * i2c_task
 *
 * Copyright (C) 2022 Texas Instruments Incorporated
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/******************************************************************************
 *
 * vI2CTask is called by main() to configure the I2C0 module for loopback mode.
 * This includes setting up the master and slave module using two tasks:
 * prvTransmitTask and prvReceiveTask.
 *
 * prvTransmitTask sends data to the I2C shift register.  After the data is
 * written, the transmit task sends a notification to the receive task
 * prvReceiveTask.
 *
 * When prvReceiveTask receives the notification, it is unblocked and then
 * reads the received data from the shift register and displays the data on
 * the terminal window.
 *
 * This example uses UARTprintf for output of UART messages.  UARTprintf is not
 * a thread-safe API and is only being used for simplicity of the demonstration
 * and in a controlled manner.
 *
 */

/* Standard includes. */
#include <stdio.h>
#include <stdint.h>
#include <stdbool.h>

/* Kernel includes. */
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"

/* Hardware includes. */
#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "driverlib/gpio.h"
#include "driverlib/i2c.h"
#include "driverlib/pin_map.h"
#include "driverlib/sysctl.h"
#include "drivers/rtos_hw_drivers.h"
#include "utils/uartstdio.h"
/*-----------------------------------------------------------*/

/*
 * Set the address for slave module. This is a 7-bit address sent in the
 * following format: [A6:A5:A4:A3:A2:A1:A0:RS]
 * A zero in the "RS" position of the first byte means that the master
 * transmits (sends) data to the selected slave, and a one in this position
 * means that the master receives data from the slave.
 */
#define SLAVE_ADDRESS 0x3C

/*
 * The notification used by the task.
 */
TaskHandle_t xTaskI2CHandle = NULL;

/*
 * The tasks as described in the comments at the top of this file.
 */
static void prvReceiveTask( void *pvParameters );
static void prvTransmitTask( void *pvParameters );

/*
 * Called by main() to create the I2C application.
 */
void vI2CTask( void );

/*
 * Hardware configuration for the I2C peripheral for loopback mode.
 */
static void prvConfigureI2C( void );
/*-----------------------------------------------------------*/

void vI2CTask( void )
{
    /* Configure the I2C0 peripheral to operate in loopback mode. */
    prvConfigureI2C();

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name for LED processing task - for debug only as it is
     *    not used by the kernel.
     *  - The size of the stack to allocate to the task.
     *  - The parameter passed to the task - just to check the functionality.
     *  - The priority assigned to the task.
     *  - The task handle used for the task notify. */
    xTaskCreate( prvReceiveTask,
                 "Rx",
                 configMINIMAL_STACK_SIZE,
                 NULL,
                 tskIDLE_PRIORITY + 2,
                 &xTaskI2CHandle );

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name for LED processing task - for debug only as it is
     *    not used by the kernel.
     *  - The size of the stack to allocate to the task.
     *  - The parameter passed to the task - just to check the functionality.
     *  - The priority assigned to the task.
     *  - The task handle used for the task notify. */
    xTaskCreate( prvTransmitTask,
                 "Tx",
                 configMINIMAL_STACK_SIZE,
                 NULL,
                 tskIDLE_PRIORITY + 1,
                 NULL );
}
/*-----------------------------------------------------------*/

static void prvTransmitTask( void *pvParameters )
{
uint32_t pui32DataTx=0;

    /* Display the example setup on the console. */
    UARTprintf("I2C Loopback Example ->");
    UARTprintf("\n   Module = I2C0");
    UARTprintf("\n   Mode = Single Send/Receive");
    UARTprintf("\n   Rate = 100kbps\n\n");
    UARTprintf("\n\r");
    UARTprintf("Tranfering from: Master -> Slave\n");

    for( ;; )
    {
        /* Display the data that the I2C0 master is transferring. */
        UARTprintf("Master Sending: %d\n", pui32DataTx % 256);

        /* Place the data to be sent in the data register. */
        I2CMasterDataPut(I2C0_BASE, pui32DataTx % 256);

        /* Initiate send of data from the master.  Since the loopback
         * mode is enabled, the master and slave units are connected
         * allowing us to receive the same data that we sent out. */
        I2CMasterControl(I2C0_BASE, I2C_MASTER_CMD_SINGLE_SEND);

        /* Wait until the slave has received and acknowledged the data. */
        while( !( I2CSlaveStatus(I2C0_BASE) & I2C_SLAVE_ACT_RREQ ) )
        {
        }

        /* Notify the receive task to process the data */
        xTaskNotifyGive(xTaskI2CHandle);

        /*increment the data to be sent */
        pui32DataTx++;

        /* wait for 10ms before sending the next data */
        vTaskDelay( pdMS_TO_TICKS( 10 ) );
    }
}
/*-----------------------------------------------------------*/

static void prvReceiveTask( void *pvParameters )
{
uint32_t pui32DataRx;

    while (1)
    {
        /* Block until the task notification is received. */
        if ( ulTaskNotifyTake( pdTRUE, portMAX_DELAY ) == pdPASS )
        {
            /* Read the data from the slave. */
            pui32DataRx = I2CSlaveDataGet(I2C0_BASE);

            /* Wait until master module is done transferring. */
            while( I2CMasterBusy(I2C0_BASE) )
            {
            }

            /* Display the data that the slave has received. */
            UARTprintf("Slave received: %d\n", pui32DataRx);
        }
    }
}
/*-----------------------------------------------------------*/

static void prvConfigureI2C( void )
{
    /* The I2C0 peripheral must be enabled before use. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_I2C0);

    /* For this example I2C0 is used with PortB[3:2].  The actual port and
     * pins used may be different on your part, consult the data sheet for
     * more information.  GPIO port B needs to be enabled so these pins can
     * be used. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOB);

    /* Configure the pin muxing for I2C0 functions on port B2 and B3.
     * This step is not necessary if your part does not support pin muxing. */
    GPIOPinConfigure(GPIO_PB2_I2C0SCL);
    GPIOPinConfigure(GPIO_PB3_I2C0SDA);

    /* Select the I2C function for these pins.  This function will also
     * configure the GPIO pins pins for I2C operation, setting them to
     * open-drain operation with weak pull-ups.  Consult the data sheet
     * to see which functions are allocated per pin. */
    GPIOPinTypeI2CSCL(GPIO_PORTB_BASE, GPIO_PIN_2);
    GPIOPinTypeI2C(GPIO_PORTB_BASE, GPIO_PIN_3);

    /* Enable loopback mode.  Loopback mode is a built in feature that is
     * useful for debugging I2C operations.  It internally connects the I2C
     * master and slave terminals, which effectively let's you send data as
     * a master and receive data as a slave.
     * NOTE: For external I2C operation you will need to use external pullups
     * that are stronger than the internal pullups.  Refer to the datasheet for
     * more information. */
    I2CLoopbackEnable(I2C0_BASE);

    /* Enable and initialize the I2C0 master module.  Use the system clock for
     * the I2C0 module.  The last parameter sets the I2C data transfer rate.
     * If false the data rate is set to 100kbps and if true the data rate will
     * be set to 400kbps.  For this example we will use a data rate of 100kbps.
     */
    I2CMasterInitExpClk(I2C0_BASE, configCPU_CLOCK_HZ, false);

    /* Enable the I2C0 slave module.  This module is enabled only for testing
     * purposes.  It does not need to be enabled for proper operation of the
     * I2C master module. */
    I2CSlaveEnable(I2C0_BASE);

    /* Set the slave address to SLAVE_ADDRESS.  In loopback mode, it's an
     * arbitrary 7-bit number (set in a macro above) that is sent to the
     * I2CMasterSlaveAddrSet function. */
    I2CSlaveInit(I2C0_BASE, SLAVE_ADDRESS);

    /* Tell the master module what address it will place on the bus when
     * communicating with the slave.  Set the address to SLAVE_ADDRESS
     * (as set in the slave module).  The receive parameter is set to false
     * which indicates the I2C Master is initiating a writes to the slave.  If
     * true, that would indicate that the I2C Master is initiating reads from
     * the slave. */
    I2CMasterSlaveAddrSet(I2C0_BASE, SLAVE_ADDRESS, false);
}
/*-----------------------------------------------------------*/

void vApplicationTickHook( void )
{
    /* This function will be called by each tick interrupt if
        configUSE_TICK_HOOK is set to 1 in FreeRTOSConfig.h.  User code can be
        added here, but the tick hook is called from an interrupt context, so
        code must not attempt to block, and only the interrupt safe FreeRTOS API
        functions can be used (those that end in FromISR()). */

    /* Only the full demo uses the tick hook so there is no code is
        executed here. */
}

